\name{MySQL}
\alias{MySQL}
\title{
  Instantiate a MySQL client from the current R or S-Plus session
}
\description{
This function creates and initializes a MySQL client.
It returns an driver object that allows you to connect
to one or several MySQL servers.
}
\usage{
MySQL(max.con = 10, fetch.default.rec = 500, force.reload = F)
}
\arguments{
\item{max.con }{
  maximum number of connections that we intended to have open.
  This can be up to 100, a limit defined at compilation time.
  Strictly speaking this limit applies to MySQL \emph{servers},
  but clients can have (at least in theory) more than this.
  }
\item{fetch.default.rec}{
  number of records to fetch at one time from the database.
  (The \code{\link[DBI]{fetch}} method uses this number as a default.)
  }
\item{force.reload}{
  should we reload (reinitialize) the client code?
  Setting this to \code{TRUE} allows you to change
  default settings.  Notice that all connections should be closed
  before re-loading.
  }
}
\value{
An object \code{MySQLDriver} that extends 
\code{dbDriver} and 
\code{dbObjectId}.
This object is required to create connections
to one or several MySQL database engines.
}
\section{Side Effects}{
The R/S-Plus client part of the database communication is initialized,
but note that connecting to the database engine needs to be done through
calls to \code{\link[DBI]{dbConnect}}.
}
\details{
This object is a singleton, that is, on subsequent invocations
it returns the same initialized object. 

This implementation allows you to connect
to multiple host servers and run multiple connections on each server
simultaneously.
}
\section{User authentication}{
The preferred method to pass authentication parameters to the
server (e.g., user, password, host) is through the MySQL personal
configuration file \file{\$HOME/.my.cnf} (or \file{c:/my.cnf} under Windows).
Since specifying passwords on calls to 
\code{dbConnect}
is a very bad idea (and so is specifying passwords through
shell variables),
the client code parses the configuration file \file{\$HOME/.my.cnf};
this file consists of zero or more sections, each starting with a 
line of the form \code{[section-name]}, for instance

\$ cat \$HOME/.my.cnf\cr
\# this is a comment\cr
[client]\cr
user = dj\cr
host = localhost\cr
\cr
[rs-dbi]\cr
database = s-data\cr
\cr
[lasers]\cr
user = opto\cr
database = opto\cr
password = pure-light\cr
host = merced\cr
...\cr
[iptraffic]\cr
host = data\cr
database = iptraffic\cr
\cr
This file should be readeable only by you. 
Inside each section, MySQL parameters may be specified one per line
(e.g., \code{user = opto}).  
The R/S MySQL implementation always parses the \code{[client]}
and \code{[rs-dbi]} sections, but you may define you own
project-specific sections to tailor its environment; 
if the same parameter appears more than once, the last
(closer to the bottom) occurrence is used.

If you define a section, say, \code{[iptraffic]},
then instead of including all these parameters in the
call to \code{dbConnect}, you simply supply the
name of the \code{group},
e.g., \code{dbConnect(mgr, group = "iptraffic")}.

The most important parameters are \code{user}, 
\code{password}, \code{host}, 
and \code{dbname}.
}
\notes{
The \code{dbname} cannot go in the
\code{[client]} section, but you may safely include 
it under the \code{[rs-dbi]} section or one you define
yourself.
}
\author{David A. James}
\section{References}{
See the Omega Project for Statistical Computing
at \url{http://www.omegahat.org}
for more details on the R/S database interface.

See the documentation at the MySQL Web site
\url{http://www.mysql.com} for details.
}
\seealso{
On database managers:

\code{\link[DBI]{dbDriver}}
\code{\link[DBI]{dbUnloadDriver}}

On connections, SQL statements and resultSets:

\code{\link[DBI]{dbConnect}}
\code{\link[DBI]{dbDisconnect}}
\code{\link[DBI]{dbSendQuery}}
\code{\link[DBI]{dbGetQuery}}
\code{\link[DBI]{fetch}}
\code{\link[DBI]{dbClearResult}}

On transaction management:

\code{\link[DBI]{dbCommit}}
\code{\link[DBI]{dbRollback}}

On meta-data:

\code{\link{summary}}
\code{\link[DBI]{dbGetInfo}}
\code{\link[DBI]{dbGetDBIVersion}}
\code{\link[DBI]{dbListTables}}
\code{\link[DBI]{dbListConnections}}
\code{\link[DBI]{dbListResults}}
\code{\link[DBI]{dbColumnInfo}}
\code{\link[DBI]{dbGetException}}
\code{\link[DBI]{dbGetStatement}}
\code{\link[DBI]{dbHasCompleted}}
\code{\link[DBI]{dbGetRowCount}}
\code{\link[DBI]{dbGetAffectedRows}}
}
\examples{\dontrun{
# create a MySQL instance and create one connection.
> m <- dbDriver("MySQL")
<MySQLDriver:(4378)>

# open the connection using user, passsword, etc., as
# specified in the "[iptraffic]" section of the 
# configuration file \file{\$HOME/.my.cnf}
> con <- dbConnect(m, group = "iptraffic")
> rs <- dbSendQuery(con, "select * from HTTP_ACCESS where IP_ADDRESS = '127.0.0.1'")
> df <- fetch(rs, n = 50)
> dbHasCompleted(rs)
[1] FALSE
> df2 <- fetch(rs, n = -1)
> dbHasCompleted(rs)
[1] TRUE
> dbClearResult(rs)
> dim(dbGetQuery(con, "show tables"))
[1] 74   1
> dbListTables(con)    
}
}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\name{MySQLConnection-class}
\docType{class}
\alias{MySQLConnection-class}
\title{Class MySQLConnection}
\description{MySQLConnection class.}
\section{Generators}
    The method \code{\link[DBI]{dbConnect}} is the main generator.
}
\section{Extends}{
Class \code{"DBIConnection"}, directly.
Class \code{"MySQLObject"}, directly.
Class \code{"DBIObject"}, by class "DBIConnection".
Class \code{"dbObjectId"}, by class "MySQLObject".
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "MySQLConnection", to = "MySQLResult")}: ... }
    \item{\link[DBI]{dbCallProc}}{\code{signature(conn = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbCommit}}{\code{signature(conn = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbConnect}}{\code{signature(drv = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbDisconnect}}{\code{signature(conn = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbExistsTable}}{\code{signature(conn = "MySQLConnection", name = "character")}: ... }
    \item{\link[DBI]{dbGetException}}{\code{signature(conn = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbGetInfo}}{\code{signature(dbObj = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbGetQuery}}{\code{signature(conn = "MySQLConnection", statement = "character")}: ... }
    \item{\link[DBI]{dbListFields}}{\code{signature(conn = "MySQLConnection", name = "character")}: ... }
    \item{\link[DBI]{dbListResults}}{\code{signature(conn = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbListTables}}{\code{signature(conn = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbReadTable}}{\code{signature(conn = "MySQLConnection", name = "character")}: ... }
    \item{\link[DBI]{dbRemoveTable}}{\code{signature(conn = "MySQLConnection", name = "character")}: ... }
    \item{\link[DBI]{dbRollback}}{\code{signature(conn = "MySQLConnection")}: ... }
    \item{\link[DBI]{dbSendQuery}}{\code{signature(conn = "MySQLConnection", statement = "character")}: ... }
    \item{\link[DBI]{dbWriteTable}}{\code{signature(conn = "MySQLConnection", name = "character", value = "data.frame")}: ... }
    \item{summary}{\code{signature(object = "MySQLConnection")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}

\seealso{
 DBI classes:
 \code{\link{MySQLObject-class}}
 \code{\link{MySQLDriver-class}}
 \code{\link{MySQLConnection-class}}
 \code{\link{MySQLResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("MySQL)
con <- dbConnect(drv, dbname = "rsdbi.db")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\name{MySQLDriver-class}
\docType{class}
\alias{MySQLDriver-class}
\title{Class MySQLDriver}
\description{
   An MySQL driver implementing the R/S-Plus database (DBI) API.
}
\section{Generators}{
   The main generators are \code{\link[DBI]{dbDriver}} and 
   \code{\link{MySQL}}.
}
\section{Extends}{
Class \code{"DBIDriver"}, directly.
Class \code{"MySQLObject"}, directly.
Class \code{"DBIObject"}, by class "DBIDriver".
Class \code{"dbObjectId"}, by class "MySQLObject".
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "MySQLObject", to = "MySQLDriver")}: ... }
    \item{\link[DBI]{dbConnect}}{\code{signature(drv = "MySQLDriver")}: ... }
    \item{\link[DBI]{dbGetInfo}}{\code{signature(dbObj = "MySQLDriver")}: ... }
    \item{\link[DBI]{dbListConnections}}{\code{signature(drv = "MySQLDriver")}: ... }
    \item{\link[DBI]{dbUnloadDriver}}{\code{signature(drv = "MySQLDriver")}: ... }
    \item{\link{summary}}{\code{signature(object = "MySQLDriver")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}

\seealso{
 DBI classes:
 \code{\link{MySQLObject-class}}
 \code{\link{MySQLDriver-class}}
 \code{\link{MySQLConnection-class}}
 \code{\link{MySQLResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("MySQL")
con <- dbConnect(drv, "user/password@dbname")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\name{MySQLObject-class}
\docType{class}
\alias{MySQLObject-class}
\title{Class MySQLObject}
\description{
   Base class for all MySQL-specific DBI classes
}
\section{Objects from the Class}{
   A virtual Class: No objects may be created from it.
}
\section{Extends}{
Class \code{"DBIObject"}, directly.
Class \code{"dbObjectId"}, directly.
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "MySQLObject", to = "MySQLriver")}: ... }
    \item{\link[DBI]{dbDataType}}{\code{signature(dbObj = "MySQLObject")}: ... }
    \item{\link[DBI]{isSQLKeyword}}{\code{signature(dbObj = "MySQLObject", name = "character")}: ... }
    \item{\link[DBI]{make.db.names}}{\code{signature(dbObj = "MySQLObject", snames = "character")}: ... }
    \item{\link[DBI]{SQLKeywords}}{\code{signature(dbObj = "MySQLObject")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}

\seealso{
 DBI classes:
 \code{\link{MySQLObject-class}}
 \code{\link{MySQLDriver-class}}
 \code{\link{MySQLConnection-class}}
 \code{\link{MySQLResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("MySQL")
con <- dbConnect(drv, dbname = "rsdbi.db")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\name{MySQLResult-class}
\docType{class}
\alias{MySQLResult-class}
\title{Class MySQLResult}
\description{
   MySQL's query results class.  This classes encapsulates the
   result of an SQL statement (either \code{select} or not).
}
\section{Generators}{
   The main generator is \code{\link[DBI]{dbSendQuery}}.
}
\section{Extends}{
Class \code{"DBIResult"}, directly.
Class \code{"MySQLObject"}, directly.
Class \code{"DBIObject"}, by class "DBIResult".
Class \code{"dbObjectId"}, by class "MySQLObject".
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "MySQLConnection", to = "MySQLResult")}: ... }
    \item{\link[DBI]{dbClearResult}}{\code{signature(res = "MySQLResult")}: ... }
    \item{\link[DBI]{dbColumnInfo}}{\code{signature(res = "MySQLResult")}: ... }
    \item{\link[DBI]{dbGetException}}{\code{signature(conn = "MySQLResult")}: ... }
    \item{\link[DBI]{dbGetInfo}}{\code{signature(dbObj = "MySQLResult")}: ... }
    \item{\link[DBI]{dbGetRowCount}}{\code{signature(res = "MySQLResult")}: ... }
    \item{\link[DBI]{dbGetRowsAffected}}{\code{signature(res = "MySQLResult")}: ... }
    \item{\link[DBI]{dbGetStatement}}{\code{signature(res = "MySQLResult")}: ... }
    \item{\link[DBI]{dbHasCompleted}}{\code{signature(res = "MySQLResult")}: ... }
    \item{\link[DBI]{dbListFields}}{\code{signature(conn = "MySQLResult", name = "missing")}: ... }
    \item{\link[DBI]{fetch}}{\code{signature(res = "MySQLResult", n = "numeric")}: ... }
    \item{\link[DBI]{fetch}}{\code{signature(res = "MySQLResult", n = "missing")}: ... }
    \item{\link{summary}}{\code{signature(object = "MySQLResult")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}

\seealso{
 DBI classes:
 \code{\link{MySQLObject-class}}
 \code{\link{MySQLDriver-class}}
 \code{\link{MySQLConnection-class}}
 \code{\link{MySQLResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("MySQL")
con <- dbConnect(drv, dbname = "rsdbi.db")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\name{S4R}
\alias{ErrorClass}
\alias{usingR}

\title{R compatibility with S version 4/Splus5+ support functions}
\description{
  These objects ease the task of porting functions into R
  from S Version 4 and S-Plus 5.0 and later.  See the documentation
  there. May be obsolete in the future.
}
\usage{
usingR(major, minor)
}
\example{\dontrun{
rc <- try(fetch(res, n = -1))
if(inherit(rc, ErrorClass))
   stop("could not fetch the data")
}
}
\keyword{internal}
% vim:syntax=tex
\name{dbApply}
\alias{dbApply}
\alias{dbApply.default}
\title{Apply R/S-Plus functions to remote groups of DBMS rows (experimental)}
\description{
  Applies R/S-Plus functions to groups of remote DBMS rows without
  bringing an entire result set all at once.  The result set
  is expected to be sorted by the grouping field.
}
\usage{
dbApply(res, ...)
dbApply.default(res, ...)
}
\arguments{
  \item{res}{a result set (see \code{\link[DBI]{dbSendQuery}}).}
  \item{...}{any additional arguments to be passed to \code{FUN}.}
}
\details{
  \code{dbApply} 
  This generic is meant to handle somewhat gracefully(?) large amounts 
  of data from the DBMS by bringing into R manageable chunks;
  the idea is that the data from individual groups can be handled by R, but
  not all the groups at the same time.  

  Currently, only the \code{\link{MySQL}} driver implements a method 
  (see the helper function \code{\link{mysqlDBApply}}) for this 
  generic function.
}
\value{
  A list with as many elements as there were groups in the
  result set.
}

\seealso{
   \code{\link{MySQL}}
   \code{\link{mysqlDBApply}}
   \code{\link[DBI]{dbSendQuery}}
   \code{\link[DBI]{fetch}}
}

\examples{\dontrun{
## compute quanitiles for each network agent
con <- dbConnect(MySQL(), group="vitalAnalysis")
rs <- dbSendQuery(con, 
             "select Agent, ip_addr, DATA from pseudo_data order by Agent")
out <- dbApply(rs, INDEX = "Agent", 
        FUN = function(x, grp) quantile(x$DATA, names=FALSE))
}
}
\keyword{programming}% at least one, from doc/KEYWORDS
\keyword{interface}% __ONLY ONE__ keyword per line
\keyword{database}
% vim: syntax=tex
\name{dbObjectId-class}
\docType{class}
\alias{dbObjectId-class}
\title{Class dbObjectId}
\description{  
   A helper (mixin) class to provide external references in
   an R/S-Plus portable way.
}
\section{Objects from the Class}{A virtual Class: No objects may be created from it.}
\section{Slots}{
  \describe{
    \item{\code{Id}:}{Object of class \code{"integer"} 
    this is an integer vector holding an opaque reference into a C struct
    (may or may not be a C pointer, may or may not have length one).
    }
  }
}

\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "dbObjectId", to = "integer")}: ... }
    \item{\link{coerce}}{\code{signature(from = "dbObjectId", to = "numeric")}: ... }
    \item{\link{coerce}}{\code{signature(from = "dbObjectId", to = "character")}: ... }
    \item{\link{format}}{\code{signature(x = "dbObjectId")}: ... }
    \item{\link{print}}{\code{signature(x = "dbObjectId")}: ... }
    \item{\link{show}}{\code{signature(object = "dbObjectId")}: ... }
  }
}

\note{A cleaner mechanism would use external references, but
historically this class has existed mainly for R/S-Plus portability.}

\examples{\dontrun{
  pg <- dbDriver("PostgreSQL")
  con <- dbConnect(pg, "user", "password")
  is(pg, "dbObjectId")   ## True
  is(con, "dbObjectId")  ## True
  isIdCurrent(con)       ## True
  q("yes")
  \$ R 
  isIdCurrent(con)       ## False
}
}
\keyword{classes}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\name{isIdCurrent}
\alias{isIdCurrent}
\title{
  Check whether a database handle object is valid or not
}
\description{
Support function that verifies that an object holding a reference
to a foreign object is still valid for communicating with the RDBMS
}
\usage{
isIdCurrent(obj)
}
\arguments{
\item{obj}{
any \code{dbObject} (e.g., \code{dbDriver},
\code{dbConnection}, \code{dbResult}).
}
}
\value{
a logical scalar.
}
\details{
\code{dbObjects} are R/S-Plus remote references to foreign objects.
This introduces differences to the object's semantics such as
persistence (e.g., connections may be closed unexpectedly),
thus this function provides a minimal verification to ensure 
that the foreign object being referenced can be contacted.
}
\seealso{
\code{\link[DBI]{dbDriver}}
\code{\link[DBI]{dbConnect}}
\code{\link[DBI]{dbSendQuery}}
\code{\link[DBI]{fetch}}
}
\examples{\dontrun{
cursor <- dbSendQuery(con, sql.statement)
isIdCurrent(cursor)
}
}
\keyword{interface}
\keyword{database}
% docclass is function
% vim: syntax=tex
\name{mysqlDBApply}
\alias{mysqlDBApply}
\title{Apply R/S-Plus functions to remote groups of DBMS rows (experimental)}
\description{
  Applies R/S-Plus functions to groups of remote DBMS rows without
  bringing an entire result set all at once.  The result set
  is expected to be sorted by the grouping field.
}
\usage{
mysqlDBApply(res, INDEX, FUN, 
   begin, group.begin, new.record, end,
   batchSize = 100, maxBatch = 1e5, ..., 
   simplify = FALSE)
}
\arguments{
  \item{res}{a result set (see \code{\link[DBI]{dbSendQuery}}).}
  \item{INDEX}{a character or integer specifying the field name or
               field number that defines the various groups.}
  \item{FUN}{a function to be invoked upon identifying the last
               row from every group. This function will be passed
               a data frame holding the records of the current group, 
               a character string with the group label, plus any
               other arguments passed to \code{dbApply} as \code{"..."}.}
  \item{begin}{a function of no arguments to be invoked just prior to 
               retrieve the first row from the result set.}
  \item{end}{a function of no arguments to be invoked just after retrieving 
               the last row from the result set.}
  \item{group.begin}{a function of one argument (the group label) to be 
               invoked upon identifying a row from a new group}.
  \item{new.record}{a function to be invoked as each individual record
               is fetched.  The first argument to this function is a
               one-row data.frame holding the new record.}
  \item{batchSize}{the default number of rows to bring from the remote 
                result set. If needed, this is automatically extended
                to hold groups bigger than \code{batchSize}.}
  \item{maxBatch}{the absolute maximum of rows per group that may
                be extracted from the result set.}
  \item{...}{any additional arguments to be passed to \code{FUN}.}
  \item{simplify}{Not yet implemented}
}
\details{
  \code{dbApply} 
  This function is meant to handle somewhat gracefully(?) large amounts 
  of data from the DBMS by bringing into R manageable chunks (about 
  \code{batchSize} records at a time, but not more than \code{maxBatch}); 
  the idea is that the data from individual groups can be handled by R, but
  not all the groups at the same time.  
  
  The MySQL implementation \code{mysqlDBApply} allows us to register R 
  functions that get invoked
  when certain fetching events occur. These include the ``begin'' event
  (no records have been yet fetched), ``begin.group'' (the record just 
  fetched belongs to a new group), ``new record'' (every fetched record
  generates this event), ``group.end'' (the record just fetched was the
  last row of the current group), ``end'' (the very last record from the
  result set). Awk and perl programmers will find this paradigm very
  familiar (although SAP's ABAP language is closer to what we're doing).
}
\value{
  A list with as many elements as there were groups in the
  result set.
}
\note{This is an experimental version implemented only in R (there are
plans, time permitting, to implement it in S-Plus).

The terminology that we're using is closer to SQL than R.  In R
what we're referring to ``groups'' are the individual levels of
a factor (grouping field in our terminology).
}

\seealso{\code{\link{MySQL}}, \code{\link[DBI]{dbSendQuery}}, \code{\link[DBI]{fetch}}.}

\examples{\dontrun{
## compute quanitiles for each network agent
con <- dbConnect(MySQL(), group="vitalAnalysis")
res <- dbSendQuery(con, 
             "select Agent, ip_addr, DATA from pseudo_data order by Agent")
out <- dbApply(res, INDEX = "Agent", 
        FUN = function(x, grp) quantile(x$DATA, names=FALSE))
}
}
\keyword{programming}% at least one, from doc/KEYWORDS
\keyword{interface}% __ONLY ONE__ keyword per line
\keyword{database}
% vim: syntax=tex
\name{mysqlSupport}
\alias{mysqlInitDriver}            % driver-related function
\alias{mysqlDriverInfo}
\alias{mysqlDescribeDriver}
\alias{mysqlCloseDriver}           % connection-related 
\alias{mysqlNewConnection}
\alias{mysqlCloneConnection}
\alias{mysqlConnectionInfo}
\alias{mysqlDescribeConnection}
\alias{mysqlCloseConnection}
\alias{mysqlExecStatement}         % result-related
\alias{mysqlFetch}
\alias{mysqlQuickSQL}
\alias{mysqlResultInfo}
\alias{mysqlDescribeResult}
\alias{mysqlCloseResult}
\alias{mysqlDescribeFields}
\alias{mysqlReadTable}
\alias{mysqlWriteTable}            % convenience functions
\alias{mysqlDataType}
\alias{.MySQLPkgName}              % constants
\alias{.MySQLPkgVersion}
\alias{.MySQLPkgRCS}
\alias{.MySQL.NA.string}
\alias{.MySQLSQLKeywords}
\alias{.conflicts.OK}
%\non_function{}
\title{Support Functions}
\description{
   These functions are the workhorse behind the RMySQL package, but
   users need not invoke these directly.  For details see \code{\link{MySQL}}.
}
\usage{
   ## MySQLDriver-related
   mysqlInitDriver(max.con=10, fetch.default.rec = 500, force.reload=FALSE)
   mysqlDriverInfo(obj, what, ...)
   mysqlDescribeDriver(obj, verbose = FALSE, ...)
   mysqlCloseDriver(drv, ...)

   ## MySQLConnection-related
   mysqlNewConnection(drv, dbname, username, password, host, unix.socket,
         port, client.flag, groups)
   mysqlCloneConnection(con, ...)
   mysqlConnectionInfo(obj, what, ...)
   mysqlDescribeConnection(obj, verbose = FALSE, ...)
   mysqlCloseConnection(con, ...)

   ## MySQLResult-related
   mysqlExecStatement(con, statement)
   mysqlFetch(res, n=0, ...)
   mysqlQuickSQL(con, statement)
   mysqlResultInfo(obj, what, ...)
   mysqlDescribeResult(obj, verbose = FALSE, ...)
   mysqlCloseResult(res, ...)
   mysqlDescribeFields(res, ...)

   ## data mappings, convenience functions, and extensions
   mysqlDataType(obj, ...)
   mysqlReadTable(con, name, row.names = "row.names", check.names = TRUE, ...)
   mysqlWriteTable(con, name, value, field.types, row.names = TRUE, 
      overwrite=FALSE, append=FALSE, ...)
}
\arguments{
\item{max.con}{
   positive integer specifying maximum number of open connections.
   The current default of 10 is hardcoded in the C code.
}
\item{fetch.default.rec}{
   default number of rows to fetch (move to R/S-Plus).  This default is
   used in \code{mysqlFetch}. 
   The default is 500.
}
\item{force.reload}{
   logical indicating whether to re-initialize the driver.  This may be
   useful if you want to change the defaults (e.g., \code{fetch.default.rec}).
   Note that the driver is a singleton (subsequent inits just returned the
   previously initialized driver, thus this argument).
}
\item{obj}{
   any of the MySQL DBI objects (e.g., \code{MySQLConnection},
   \code{MySQLResult}).
}
\item{what}{
   character vector of metadata to extract, e.g., "version", "statement",
   "isSelect".
}
\item{verbose}{
   logical controlling how much information to display.
   Defaults to \code{FALSE}.
}
\item{drv}{
   an \code{MySQLDriver} object as produced by \code{mysqlInitDriver}.
}
\item{con}{
   an \code{MySQLConnection} object as produced by \code{mysqlNewConnection}
   and \code{mysqlCloneConnection}.
}
\item{res}{
   an \code{MySQLResult} object as produced by by \code{mysqlExecStatement}.
}
\item{username}{
   a character string with the MySQL's user name.  
}
\item{password}{
   character string with the MySQL's password.
}
\item{groups}{
   character vector with one or more MySQL group names.  For details
   see \code{\link{MySQL}}.
}
\item{dbname}{
   character string with the MySQL database name.
}
\item{host}{
   character string with the name (or IP address) of the machine
   hosting the database. Default is \code{""}, which is
   interpreted as \code{localhost} by the MySQL's API.
}
\item{unix.socket}{
   (optional) character string with a filename for the socket
   file name.  Consult the MySQL documentation for details.
}
\item{port}{
   (optional) positive integer specifying the TCP port number that
   the MySQL server is listening to.  Consult the MySQL documentation 
   for details.
}
\item{client.flag}{
   (optional) integer setting flags for the client.  Consult the
   MySQL documentation for details.
}
\item{max.results}{
   positive integer indicating the maximum number of results that
   MySQL connections will hold open.  
}
\item{force}{
   logical indicating whether to close a connection that has open
   result sets.  The default is \code{FALSE}.
}
\item{statement}{
   character string holding one (and only one) SQL statement.
}
\item{n}{
   number of rows to fetch from the given result set. A value of -1
   indicates to retrieve all the rows.  The default of 0 specifies
   to extract whatever the \code{fetch.default.rec} was specified
   during driver initialization \code{mysqlInit}.
}
\item{name}{
   character vector of names (table names, fields, keywords).
}
\item{value}{
   a data.frame.
}
\item{field.types}{
   a list specifying the mapping from R/S-Plus fields in the data.frame
   \code{value} to SQL data types.  The default is
   \code{sapply(value,SQLDataType)}, see \code{MySQLSQLType}.
}
\item{row.names}{
   a logical specifying whether to prepend the \code{value} data.frame 
   row names or not.  The default is \code{TRUE}.
}
\item{check.names}{
   a logical specifying whether to convert DBMS field names into
   legal S names. Default is \code{TRUE}.
}
\item{overwrite}{
   logical indicating whether to replace the table \code{name} with
   the contents of the data.frame \code{value}.
   The defauls is \code{FALSE}.
}
\item{append}{
   logical indicating whether to append \code{value} to the existing
   table \code{name}.
}
\item{\dots}{
   placeholder for future use.
}
}
\value{
   \code{mysqlInitDriver} returns an \code{MySQLDriver} object.

   \code{mysqlDriverInfo} returns a list of name-value metadata pairs.

   \code{mysqlDescribeDriver} returns \code{NULL} (displays the object's 
   metadata).

   \code{mysqlCloseDriver} returns a logical indicating whether the 
   operation succeeded or not.

   \code{mysqlNewConnection} returns an \code{MySQLConnection} object.

   \code{mysqlCloneConnection} returns an \code{MySQLConnection} object.

   \code{mysqlConnectionInfo}returns a list of name-value metadata pairs.

   \code{mysqlDescribeConnection} returns \code{NULL} (displays the 
   object's metadata).

   \code{mysqlCloseConnection} returns a logical indicating whether the 
   operation succeeded or not.

   \code{mysqlExecStatement} returns an \code{MySQLResult} object.

   \code{mysqlFetch} returns a data.frame.

   \code{mysqlQuickSQL} returns either a data.frame if the \code{statement} is
   a \code{select}-like or NULL otherwise.

   \code{mysqlDescribeResult} returns \code{NULL} (displays the object's 
   metadata).
   
   \code{mysqlCloseResult} returns a logical indicating whether the 
   operation succeeded or not.
   
   \code{mysqlDescribeFields} returns a data.frame with one row per field
   with columns \code{name}, \code{Sclass}, \code{type}, \code{len}, 
   \code{precision}, \code{scale}, and \code{nullOK} which fully describe
   each field in a result set.  Except for \code{Sclass} (which shows the
   mapping of the field type into an R/S-Plus class) all the information
   pertains to MySQL's data storage attributes.

   \code{mysqlReadTable}  returns a data.frame with the contents of the
   DBMS table.
   
   \code{mysqlWriteTable}  returns a logical indicating whether the 
   operation succeeded or not.
   
   \code{mysqlDataType} retuns a character string with the closest
   
   \code{mysqlResultInfo} returns a list of name-value metadata pairs.
}
\section{Constants}{
\code{.MySQLPkgName} (currently \code{"RMySQL"}),
\code{.MySQLPkgVersion} (the R package version),
\code{.MySQLPkgRCS} (the RCS revision),
\code{.MySQL.NA.string} (character that MySQL uses to 
denote \code{NULL} on input),
\code{.MySQLSQLKeywords} (a lot!)
\code{.conflicts.OK}.
%\non_function{}
}
\keyword{datasets}
\keyword{interface}
\keyword{database}
%\keyword{internal}
% vim:syntax=tex
\name{safe.write}
\alias{safe.write}
\title{Write a data.frame avoiding exceeding memory limits}
\description{
   This function batches calls to \code{write.table} to avoid
   exceeding memory limits for very large data.frames.
}
\usage{
safe.write(value, file, batch, ...)
}
\arguments{
  \item{value}{a data.frame;}
  \item{file}{a file object (connection, file name, etc).}
  \item{batch}{maximum number of rows to write at a time.}
  \item{\dots}{any other arguments are passed to \code{write.table}.}
}
\details{
   The function has a while loop invoking \code{\link[base]{write.table}}
   for subsets of \code{batch} rows of \code{value}.  Since this is
   a helper function for \code{\link[RMySQL]{mysqlWriteTable}} has hardcoded
   other arguments to \code{write.table}.
}
\value{
    \code{NULL}, invisibly.
}
\note{No error checking whatsoever is done.}

\seealso{\code{\link[base]{write.table}}} 

\examples{\dontrun{
   ctr.file <- file("dump.sqloader", "w")
   safe.write(big.data, file = ctr.file, batch = 25000)
}
}
\keyword{internal}
%% vim: syntax=tex
